#include <stdint.h>

#include "stm32wlxx_hal.h"

#include "lora_radio.h"
#include "lora_radio_private_defs.h"
#include "rf_switch.h"

/* Lora info:
 *
 * https://blog.classycode.com/lora-sync-word-compatibility-between-sx127x-and-sx126x-460324d1787a
 *
 */

/* ST info
 *
 * https://www.st.com/resource/en/reference_manual/rm0453-stm32wl5x-advanced-armbased-32bit-mcus-with-subghz-radio-solution-stmicroelectronics.pdf
 * https://www.st.com/resource/en/datasheet/stm32wle5c8.pdf
 * https://www.st.com/resource/en/errata_sheet/es0500-stm32wl55xx-stm32wl54xx-device-errata-stmicroelectronics.pdf
 * https://www.st.com/resource/en/user_manual/um2592-stm32wl-nucleo64-board-mb1389-stmicroelectronics.pdf
 *
 */

/* SemTech info
 *
 * DS_SX1261-2_V2_1.pdf
 *
 */

/*!
 * \brief TODO
 */
typedef struct
{
	loraRadio_spreadingFactors_t spreading_factor;
	loraRadio_bandwidths_t bandwidth;
	loraRadio_codingRates_t coding_rate;
	loraRadio_lowDataRateOptimization_t low_data_rate_opt;
	uint32_t channel_freq;
} loraRadio_modulationParameters_t;

/*!
 * \brief TODO
 */
typedef struct
{
	loraRadio_packetTypes_t packet_type;
	uint16_t preamble_len;
	loraRadio_headerType_t header_type;
	loraRadio_crcModes_t crc_mode;
	loraRadio_iqModes_t iq_mode;
} loraRadio_packetParameters_t;

/*!
 * \brief TODO
 */
typedef struct
{
	loraRadio_modulationParameters_t modulation_parameters;
	loraRadio_packetParameters_t packet_parameters;
	int8_t tx_power_dBm;
	loraRadio_txModes_t tx_mode;
	uint32_t tx_timeout;
} loraRadio_txParameters_t;

/*!
 * \brief TODO
 */
typedef struct
{
	loraRadio_modulationParameters_t modulation_parameters;
	loraRadio_packetParameters_t packet_parameters;
	uint8_t symbol_timeout; // TODO: Document --> used to configure the number of LoRa symbols to be received before starting the reception of a LoRa packet
	loraRadio_rxModes_t rx_mode;
	loraRadio_rxPowerMode_t rx_power_mode;
	uint32_t rx_timeout;
	uint32_t listening_mode_rx_time;
	uint32_t listening_mode_sleep_time;
	uint32_t rx_duty_cycle_preamble_detect_timeout; /*TODO: document --> See STM32WL Errata: RadioSetRxDutyCycle*/
} loraRadio_rxParameters_t;

/* External variables --------------------------------------------------------*/
// TODO:Try tyo avoid global variables in STM32CubeMX code generation
extern SUBGHZ_HandleTypeDef hsubghz; // handler generated by st api drivers
// static variables
static loraRadio_events_t loraRadio_events;
static loraRadio_txParameters_t loraRadio_txParameters;
static loraRadio_rxParameters_t loraRadio_rxParameters;

// HAL_SUBGHz Callbacks re-definitions. override the default callback of stm32wlxx_hal_subghz.c

/**
 * @brief  Packet transmission completed callback.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_TxCpltCallback(SUBGHZ_HandleTypeDef *hsubghz)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;
}

/**
 * @brief  Packet received callback.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_RxCpltCallback(SUBGHZ_HandleTypeDef *hsubghz)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;
}

/**
 * @brief  Preamble Detected callback.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_PreambleDetectedCallback(SUBGHZ_HandleTypeDef *hsubghz)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;

	// Manage the following
	/*See STM32WL Errata: RadioSetRxDutyCycle*/
}

/**
 * @brief  Valid sync word detected callback.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_SyncWordValidCallback(SUBGHZ_HandleTypeDef *hsubghz)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;
}

/**
 * @brief  Valid LoRa header received callback.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_HeaderValidCallback(SUBGHZ_HandleTypeDef *hsubghz)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;
}

/**
 * @brief  LoRa header CRC error callback.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_HeaderErrorCallback(SUBGHZ_HandleTypeDef *hsubghz)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;
}

/**
 * @brief  Wrong CRC received callback.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_CRCErrorCallback(SUBGHZ_HandleTypeDef *hsubghz)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;
}

/**
 * @brief  Channel activity detection status callback.
 * @note   Unified callback for CAD Done and CAD activity interrupts.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @param  cadstatus reports whether activity is detected or not
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_CADStatusCallback(SUBGHZ_HandleTypeDef *hsubghz,
																												HAL_SUBGHZ_CadStatusTypeDef cadstatus)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;
	(void)cadstatus;
}

/**
 * @brief  Rx or Tx Timeout callback.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_RxTxTimeoutCallback(SUBGHZ_HandleTypeDef *hsubghz)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;
}

/**
 * @brief  LR FHSS Hop callback.
 * @param  hsubghz pointer to a SUBGHZ_HandleTypeDef structure that contains
 *                 the configuration information for SUBGHZ module.
 * @retval None
 */
void __attribute__((weak)) HAL_SUBGHZ_LrFhssHopCallback(SUBGHZ_HandleTypeDef *hsubghz)
{
	/* Prevent unused argument(s) compilation warning */
	(void)hsubghz;
}

// TODO: Remove all weak attributes in callbacks

/*!
 * \brief TODO
 */
static void loraRadio_writeRegister(uint16_t address, uint8_t value)
{
	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_WriteRegisters(&hsubghz, address, &value, 1);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static uint8_t loraRadio_readRegister(uint32_t address)
{
	uint8_t value;

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ReadRegisters(&hsubghz, address, &value, 1);
	SUBGHZ_CRITICAL_SECTION_END();

	return value;
}

/*!
 * \brief TODO
 */
static void loraRadio_setStandbyCommand(loraRadio_standbyModes_t standby_mode)
{
	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_STANDBY, (uint8_t *)(&standby_mode), 1);
	SUBGHZ_CRITICAL_SECTION_END();

	if (standby_mode == STDBY_RC)
	{
		// TODO save OperatingMode = MODE_STDBY_RC; or set in the funcion call?
	}
	else //(standbyConfig == STDBY_XOSC)
	{
		// TODO set OperatingMode = MODE_STDBY_XOSC;  or set in the funcion call?
	}
}

/*!
 * \brief TODO
 */
static void loraRadio_setTxContinuousWaveCommand(void)
{
	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_TXCONTINUOUSWAVE, NULL, 0);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setSleepCommand(void)
{
	uint8_t smpsc2r_reg;
	uint8_t sleep_command_param;

	// Set RF switch OFF
	rfSwitch_setState(SWITCH_OFF);

	// Set SMPS maximum drive capability to 40mA
	smpsc2r_reg = loraRadio_readRegister(SUBGHZ_SMPSC2R);
	smpsc2r_reg &= (~SMPS_DRV_MASK);
	smpsc2r_reg |= SMPS_DRV_40_mA;
	loraRadio_writeRegister(SUBGHZ_SMPSC2R, smpsc2r_reg);

	// Set sleep command parameters
	sleep_command_param = SLEEP_CFG_START_WARM | SLEEP_CFG_RTC_WAKE_UP_DISABLED;

	// Execute Set_Sleep() command
	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_SLEEP, &sleep_command_param, 1);
	SUBGHZ_CRITICAL_SECTION_END();

	// TODO: Manage an static variable that saves the module status?
}

/*!
 * \brief TODO
 */
static void loraRadio_setTCXOModeCommand(loraRadio_tcxoCtrlVoltage_t reg_tcxo_trim, uint32_t timeout)
{
	uint8_t tcxo_mode_parameters[4];

	tcxo_mode_parameters[0] = ((uint8_t)reg_tcxo_trim) & 0x07;
	tcxo_mode_parameters[1] = (uint8_t)((timeout >> 16) & 0x000000FF);
	tcxo_mode_parameters[2] = (uint8_t)((timeout >> 8) & 0x000000FF);
	tcxo_mode_parameters[3] = (uint8_t)((timeout) & 0x000000FF);

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_TCXOMODE, tcxo_mode_parameters, 4);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_calibrateCommand(uint8_t calibration_mask)
{
	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_CALIBRATE, &calibration_mask, 1);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setPacketTypeCommand(uint8_t packet_type)
{
	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_PACKETTYPE, &packet_type, 1);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setModulationParamsCommand(loraRadio_spreadingFactors_t spreading_factor,
																								 loraRadio_bandwidths_t bandwidth,
																								 loraRadio_codingRates_t coding_rate,
																								 loraRadio_lowDataRateOptimization_t low_data_rate_opt)
{
	uint8_t set_modulation_parameters[4];

	set_modulation_parameters[0] = spreading_factor;
	set_modulation_parameters[1] = bandwidth;
	set_modulation_parameters[2] = coding_rate;
	set_modulation_parameters[3] = low_data_rate_opt;

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_MODULATIONPARAMS, set_modulation_parameters, 4);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setPacketParamsCommand(uint16_t preamble_len,
																						 loraRadio_headerType_t header_type,
																						 uint8_t len,
																						 loraRadio_crcModes_t crc_mode,
																						 loraRadio_iqModes_t iq_mode)
{
	uint8_t packet_parameters[6];

	// Set packet parameters
	packet_parameters[0] = (uint8_t)((preamble_len >> 8) & 0x00FF);
	packet_parameters[1] = (uint8_t)((preamble_len) & 0x00FF);
	;
	packet_parameters[2] = (uint8_t)(header_type);
	packet_parameters[3] = len;
	packet_parameters[4] = (uint8_t)(crc_mode);
	packet_parameters[5] = (uint8_t)(iq_mode);

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_PACKETPARAMS, packet_parameters, 6);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setLoraSymbTimeoutCommand(uint8_t symb_timeout)
{
	uint8_t mant = symb_timeout >> 1;
	uint8_t exp = 0;
	uint8_t reg = 0;

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_LORASYMBTIMEOUT, &symb_timeout, 1);
	SUBGHZ_CRITICAL_SECTION_END();

	if (symb_timeout >= 64)
	{

		while (mant > 31)
		{
			mant >>= 2;
			exp++;
		}

		reg = exp + (mant << 3);

		loraRadio_writeRegister(SUBGHZ_LSYNCTIMEOUTR, reg);
	}
}

/*!
 * \brief TODO
 */
static void loraRadio_setRegulatorModeCommand(loraRadio_regulatorMode_t regulator_mode)
{
	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_REGULATORMODE, (uint8_t *)(&regulator_mode), 1);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setBufferBaseAddressCommand(uint8_t tx_address, uint8_t rx_address)
{
	uint8_t buffer_addresses[2];

	buffer_addresses[0] = tx_address;
	buffer_addresses[1] = rx_address;

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_BUFFERBASEADDRESS, buffer_addresses, 2);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setTxParamsCommand(int8_t power_dBm, loraRadio_rampTimes_t ramp_time)
{
	uint8_t tx_parameters[2];

	tx_parameters[0] = (uint8_t)power_dBm;
	tx_parameters[1] = ramp_time;

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_TXPARAMS, tx_parameters, 2);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setTxCommand(uint32_t timeout)
{
	uint8_t set_tx_parameters[3];

	set_tx_parameters[0] = (uint8_t)((timeout >> 16) & 0x000000FF);
	set_tx_parameters[1] = (uint8_t)((timeout >> 8) & 0x000000FF);
	set_tx_parameters[2] = (uint8_t)(timeout & 0x000000FF);

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_TX, set_tx_parameters, 3);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setPayload(uint8_t *payload, uint8_t len)
{
	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_WriteBuffer(&hsubghz, 0x00, payload, len); // set payload
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setPaConfigCommand(uint8_t pa_duty_cycle, uint8_t hp_max, uint8_t pa_sel)
{
	uint8_t power_amplifier_config[4];

	power_amplifier_config[0] = pa_duty_cycle; // PaDutyCycle
	power_amplifier_config[1] = hp_max;				 // HpMax
	power_amplifier_config[2] = pa_sel;				 // PaSel
	power_amplifier_config[3] = LP_PA_LUT;		 // PaLUT

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_PACONFIG, power_amplifier_config, 4);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_configDioIrqCommand(uint16_t irq_mask)
{
	uint8_t irq_masks[8];

	/* Remark
	 * The reference manual doesn't say this explicitly, but it seems that internally all interrupts can only
	 * be assigned to the IRQ1 line since it's always done that way in the ST "SubGHz_Phy" example software.
	 * If we look at DS_SX1261-2 there is the possibility to map the enabled IRQs to DIO1, DIO2 and DIO3.
	 * But I don't find how it's done internally in the SubGhz module of ST, so I always map all interrupts
	 * to the IRQ1 line.
	 *
	 * See -> https://community.st.com/t5/stm32-mcus-wireless/stm32wl-subghz-radio-irq-lines/td-p/151558
	 */
	irq_masks[0] = (uint8_t)((irq_mask >> 8) & 0x00FF);				// IRQ mask High Byte
	irq_masks[1] = (uint8_t)((irq_mask) & 0x00FF);						// IRQ mask Low Byte
	irq_masks[2] = (uint8_t)((irq_mask >> 8) & 0x00FF);				// IRQ1 line mask High Byte
	irq_masks[3] = (uint8_t)((irq_mask) & 0x00FF);						// IRQ1 line mask Low Byte
	irq_masks[4] = (uint8_t)((IRQ_RADIO_NONE >> 8) & 0x00FF); // IRQ2 line mask High Byte
	irq_masks[5] = (uint8_t)((IRQ_RADIO_NONE) & 0x00FF);			// IRQ2 line mask Low Byte
	irq_masks[6] = (uint8_t)((IRQ_RADIO_NONE >> 8) & 0x00FF); // IRQ3 line mask High Byte
	irq_masks[7] = (uint8_t)((IRQ_RADIO_NONE) & 0x00FF);			// IRQ3 line mask Low Byte

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_CFG_DIOIRQ, irq_masks, 8);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_calibrateImageCommand(uint32_t freq)
{
	uint8_t calibrate_image_parameters[2];

	// cal_freq values obtained from the "CalibrateImage()" command specification. See reference manual
	if (freq > 900000000)
	{
		calibrate_image_parameters[0] = 0xE1;
		calibrate_image_parameters[1] = 0xE9;
	}
	else if (freq > 850000000)
	{
		calibrate_image_parameters[0] = 0xD7;
		calibrate_image_parameters[1] = 0xDB;
	}
	else if (freq > 770000000)
	{
		calibrate_image_parameters[0] = 0xC1;
		calibrate_image_parameters[1] = 0xC5;
	}
	else if (freq > 460000000)
	{
		calibrate_image_parameters[0] = 0x75;
		calibrate_image_parameters[1] = 0x81;
	}
	else if (freq > 425000000)
	{
		calibrate_image_parameters[0] = 0x6B;
		calibrate_image_parameters[1] = 0x6F;
	}
	else // freq <= 425000000
	{
		// [ 156MHz - 171MHz ]
		calibrate_image_parameters[0] = 0x29;
		calibrate_image_parameters[1] = 0x2B;
	}

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_CALIBRATEIMAGE, calibrate_image_parameters, 2);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setRfFrequencyCommand(uint32_t channel_freq)
{
	uint8_t set_rf_freq_parameters[4];
	uint32_t freq;

	// Calibrate for each new requested channel frequency
	loraRadio_calibrateImageCommand(channel_freq);

	// formula to obtain the freq value for the channel frequency requested. See "Set_RfFrequency()" command in reference manual
	freq = (uint32_t)((((uint64_t)channel_freq) << 25) / (XTAL_FREQ));

	set_rf_freq_parameters[0] = (uint8_t)((freq >> 24) & 0x000000FF);
	set_rf_freq_parameters[1] = (uint8_t)((freq >> 16) & 0x000000FF);
	set_rf_freq_parameters[2] = (uint8_t)((freq >> 8) & 0x000000FF);
	set_rf_freq_parameters[3] = (uint8_t)(freq & 0x000000FF);

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_RFFREQUENCY, set_rf_freq_parameters, 4);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setRxCommand(uint32_t timeout)
{
	uint8_t set_rx_parameters[3];

	set_rx_parameters[0] = (uint8_t)((timeout >> 16) & 0x000000FF);
	set_rx_parameters[1] = (uint8_t)((timeout >> 8) & 0x000000FF);
	set_rx_parameters[2] = (uint8_t)(timeout & 0x000000FF);

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_RX, set_rx_parameters, 3);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
static void loraRadio_setRxDutyCycleCommand(uint32_t rx_time, uint32_t sleep_time)
{
	uint8_t set_rx_duty_cycle_parameters[6];

	set_rx_duty_cycle_parameters[0] = (uint8_t)((rx_time >> 16) & 0x000000FF);
	set_rx_duty_cycle_parameters[1] = (uint8_t)((rx_time >> 8) & 0x000000FF);
	set_rx_duty_cycle_parameters[2] = (uint8_t)(rx_time & 0x000000FF);
	set_rx_duty_cycle_parameters[3] = (uint8_t)((sleep_time >> 16) & 0x000000FF);
	set_rx_duty_cycle_parameters[4] = (uint8_t)((sleep_time >> 8) & 0x000000FF);
	set_rx_duty_cycle_parameters[5] = (uint8_t)(sleep_time & 0x000000FF);

	SUBGHZ_CRITICAL_SECTION_BEGIN();
	HAL_SUBGHZ_ExecSetCmd(&hsubghz, RADIO_SET_RXDUTYCYCLE, set_rx_duty_cycle_parameters, 6);
	SUBGHZ_CRITICAL_SECTION_END();
}

/*!
 * \brief TODO
 */
void loraRadio_init(void)
{
	uint8_t smpsc2r_reg;

	hsubghz.Init.BaudratePrescaler = SUBGHZSPI_BAUDRATEPRESCALER_4;
	if (HAL_SUBGHZ_Init(&hsubghz) != HAL_OK)
	{
		// TODO: Manage an Error_Handler() ?
	}

	// Initialize RF switch
	rfSwitch_init();

	// Go standby
	loraRadio_setStandbyCommand(STDBY_RC);

	// Set SMPS maximum drive capability to 40mA
	smpsc2r_reg = loraRadio_readRegister(SUBGHZ_SMPSC2R);
	smpsc2r_reg &= (~SMPS_DRV_MASK);
	smpsc2r_reg |= SMPS_DRV_40_mA;
	loraRadio_writeRegister(SUBGHZ_SMPSC2R, smpsc2r_reg);

	// Initialize Temperature compensated Crystal oscillator (TCXO) control
	loraRadio_setTCXOModeCommand(TCXO_CTRL_1_8V, TCXO_AND_HSE32_READY_TIMEOUT_100ms);
	loraRadio_writeRegister(SUBGHZ_HSEINTRIMR, LOAD_CAPACITOR_TRIMMING_VALUE_11_3_pF);
	loraRadio_calibrateCommand(ALL_CALIBRATION);

	/* Remark --> workaround done in ST "SubGHz_Phy" example software
	 * WORKAROUND - Trimming the output voltage power_ldo to 3.3V
	 */
	loraRadio_writeRegister(SUBGHZ_REGDRVCR, REGULATOR_DRIVE_TRIMMING);

	// set modem to LoRa bye default. For the moment it's not allowed to change by higher layers
	loraRadio_txParameters.packet_parameters.packet_type = PACKET_TYPE_LORA;
	loraRadio_rxParameters.packet_parameters.packet_type = PACKET_TYPE_LORA;
	loraRadio_setPacketTypeCommand((uint8_t)PACKET_TYPE_LORA);

	// Set LoRa radio power regulator mode
	loraRadio_setRegulatorModeCommand(SMPS);

	// Set LoRa radio tx and rx buffer address (half duplex radio, so we use the same address for both)
	loraRadio_setBufferBaseAddressCommand(0x00, 0x00);

	// Set default tx parameters
	loraRadio_setTxParamsCommand(0 + LP_POWER_OFFSET_dBm, RAMP_200_us);

	// set power amplifier to LP PA mode
	loraRadio_setPaConfigCommand(LP_PA_DUTY_CYCLE, LP_HP_MAX, LP_PA_SEL);
	loraRadio_writeRegister(SUBGHZ_PAOCPR, LP_OVER_CURRENT_PROTECTION_VALUE_FOR_PAOCPR_REGISTER);

	// Set interrupts. Enable all interrupts by default
	// TODO: Check if is mandatory to enable all Radio Interrupts or only the minimum needed.
	loraRadio_configDioIrqCommand(IRQ_RADIO_ALL);

	// Go sleep
	loraRadio_setSleepCommand();
}

/*!
 * \brief TODO
 */
void loraRadio_setListenersOnRadioEvents(loraRadio_events_t *radio_events)
{
	(void)loraRadio_events;
	(void)radio_events;

	// TODO: Set listener to events and manage them
}

/*!
 * \brief TODO
 */
void loraRadio_setTxParameters(loraRadio_spreadingFactors_t spreading_factor,
															 loraRadio_bandwidths_t bandwidth,
															 loraRadio_codingRates_t coding_rate,
															 uint32_t channel_freq,
															 uint16_t preamble_len,
															 loraRadio_headerType_t header_type,
															 loraRadio_crcModes_t crc_mode,
															 loraRadio_iqModes_t iq_mode,
															 int8_t tx_power_dBm,
															 loraRadio_txModes_t tx_mode,
															 uint32_t tx_timeout)
{
	loraRadio_txParameters.modulation_parameters.spreading_factor = spreading_factor;
	loraRadio_txParameters.modulation_parameters.bandwidth = bandwidth;
	loraRadio_txParameters.modulation_parameters.coding_rate = coding_rate;
	loraRadio_txParameters.modulation_parameters.channel_freq = channel_freq;
	loraRadio_txParameters.packet_parameters.preamble_len = preamble_len;
	loraRadio_txParameters.packet_parameters.header_type = header_type;
	loraRadio_txParameters.packet_parameters.crc_mode = crc_mode;
	loraRadio_txParameters.packet_parameters.iq_mode = iq_mode;
	loraRadio_txParameters.tx_power_dBm = tx_power_dBm;
	loraRadio_txParameters.tx_mode = tx_mode;
	loraRadio_txParameters.tx_timeout = tx_timeout;

	// TODO: verify that the specified power is within the specified range?

	// Check whether it is worth enabling LDRO. See ST "SubGHz_Phy" example software.
	if (((bandwidth == LORA_BW_007) && ((spreading_factor == LORA_SF11) || (spreading_factor == LORA_SF12))) ||
			((bandwidth == LORA_BW_015) && (spreading_factor == LORA_SF12)))
	{
		loraRadio_txParameters.modulation_parameters.low_data_rate_opt = LDRO_ENABLED;
	}
	else
	{
		loraRadio_txParameters.modulation_parameters.low_data_rate_opt = LDRO_DISABLED;
	}

	/*!
	 * Remark (See reference manual)
	 *
	 * For SF5 and SF6, due to the higher symbol rate, the minimum preamble length needed to
	 * ensure correct detection and demodulation of the received signal is 12 symbols.
	 */
	if (((spreading_factor == LORA_SF5) || (spreading_factor == LORA_SF6)) &&
			(preamble_len < 12))
	{
		loraRadio_txParameters.packet_parameters.preamble_len = 12;
	}
}

/*!
 * \brief TODO
 */
void loraRadio_setRxParameters(loraRadio_spreadingFactors_t spreading_factor,
															 loraRadio_bandwidths_t bandwidth,
															 loraRadio_codingRates_t coding_rate,
															 uint32_t channel_freq,
															 uint16_t preamble_len,
															 loraRadio_headerType_t header_type,
															 loraRadio_crcModes_t crc_mode,
															 loraRadio_iqModes_t iq_mode,
															 uint8_t symbol_timeout,
															 loraRadio_rxModes_t rx_mode,
															 loraRadio_rxPowerMode_t rx_power_mode,
															 uint32_t rx_timeout,
															 uint32_t listening_mode_rx_time, // TODO: specifiy units in all the parameters when needed, example --> resolution 15.625 μs
															 uint32_t listening_mode_sleep_time)
{
	loraRadio_rxParameters.modulation_parameters.spreading_factor = spreading_factor;
	loraRadio_rxParameters.modulation_parameters.bandwidth = bandwidth;
	loraRadio_rxParameters.modulation_parameters.coding_rate = coding_rate;
	loraRadio_rxParameters.modulation_parameters.channel_freq = channel_freq;
	loraRadio_rxParameters.packet_parameters.preamble_len = preamble_len;
	loraRadio_rxParameters.packet_parameters.header_type = header_type;
	loraRadio_rxParameters.packet_parameters.crc_mode = crc_mode;
	loraRadio_rxParameters.packet_parameters.iq_mode = iq_mode;
	loraRadio_rxParameters.symbol_timeout = symbol_timeout;
	loraRadio_rxParameters.rx_mode = rx_mode;
	loraRadio_rxParameters.rx_power_mode = rx_power_mode;
	loraRadio_rxParameters.rx_timeout = rx_timeout;
	loraRadio_rxParameters.listening_mode_rx_time = listening_mode_rx_time;
	loraRadio_rxParameters.listening_mode_sleep_time = listening_mode_sleep_time;

	// Check whether it is worth enabling LDRO. See ST "SubGHz_Phy" example software.
	if (((bandwidth == LORA_BW_007) && ((spreading_factor == LORA_SF11) || (spreading_factor == LORA_SF12))) ||
			((bandwidth == LORA_BW_015) && (spreading_factor == LORA_SF12)))
	{
		loraRadio_rxParameters.modulation_parameters.low_data_rate_opt = LDRO_ENABLED;
	}
	else
	{
		loraRadio_rxParameters.modulation_parameters.low_data_rate_opt = LDRO_DISABLED;
	}

	/*!
	 * Remark (See reference manual)
	 *
	 * For SF5 and SF6, due to the higher symbol rate, the minimum preamble length needed to
	 * ensure correct detection and demodulation of the received signal is 12 symbols.
	 */
	if (((spreading_factor == LORA_SF5) || (spreading_factor == LORA_SF6)) &&
			(preamble_len < 12))
	{
		loraRadio_rxParameters.packet_parameters.preamble_len = 12;
	}
}

/*!
 * \brief TODO
 */
void loraRadio_send(uint8_t *payload, uint8_t len)
{
	// TODO: if not all the interrupts are enabled in init funcion, enable here the tx interrupts

	loraRadio_setStandbyCommand(STDBY_RC);

	if (loraRadio_txParameters.tx_power_dBm > 15) // Use HP PA
	{
		// Set RF switch to RFO HP
		rfSwitch_setState(SWITCH_RFO_HP);

		// Set tx power
		loraRadio_setTxParamsCommand(loraRadio_txParameters.tx_power_dBm + HP_POWER_OFFSET_dBm, RAMP_200_us);

		// set power amplifier to HP PA mode
		loraRadio_setPaConfigCommand(HP_PA_DUTY_CYCLE, HP_HP_MAX, HP_PA_SEL);
		loraRadio_writeRegister(SUBGHZ_PAOCPR, HP_OVER_CURRENT_PROTECTION_VALUE_FOR_PAOCPR_REGISTER);
	}
	else // Use LP PA
	{
		// Set RF switch to RFO LP
		rfSwitch_setState(SWITCH_RFO_LP);

		// Set tx power
		loraRadio_setTxParamsCommand(loraRadio_txParameters.tx_power_dBm + LP_POWER_OFFSET_dBm, RAMP_200_us);

		// set power amplifier to LP PA mode
		loraRadio_setPaConfigCommand(LP_PA_DUTY_CYCLE, LP_HP_MAX, LP_PA_SEL);
		loraRadio_writeRegister(SUBGHZ_PAOCPR, LP_OVER_CURRENT_PROTECTION_VALUE_FOR_PAOCPR_REGISTER);
	}

	// Set modulation parameters
	loraRadio_setRfFrequencyCommand(loraRadio_txParameters.modulation_parameters.channel_freq);
	loraRadio_setModulationParamsCommand(loraRadio_txParameters.modulation_parameters.spreading_factor,
																			 loraRadio_txParameters.modulation_parameters.bandwidth,
																			 loraRadio_txParameters.modulation_parameters.coding_rate,
																			 loraRadio_txParameters.modulation_parameters.low_data_rate_opt);

	// Set packet parameters
	loraRadio_setPacketParamsCommand(loraRadio_txParameters.packet_parameters.preamble_len,
																	 loraRadio_txParameters.packet_parameters.header_type,
																	 len,
																	 loraRadio_txParameters.packet_parameters.crc_mode,
																	 loraRadio_txParameters.packet_parameters.iq_mode);
	loraRadio_setPayload(payload, len);

	if (loraRadio_txParameters.tx_mode == TX_SINGLE_MODE)
	{
		loraRadio_setTxCommand((uint32_t)0);
	}
	else //(loraRadio_txParameters.tx_mode == TX_SINGLE_WITH_TIMEOUT_MODE)
	{
		loraRadio_setTxCommand(loraRadio_txParameters.tx_timeout);
	}

	// TODO: Consider setting the variable that holds the module's status to TX state?
	// Alternatively, implement this module without its own state and add a "get state" command for the SubGhz module.

	// TODO: In the interrupt handler for TX done, error, or timeout, set the radio to sleep.

	// TODO: There is a register (SUBGHZ_LPLDLENR) that serves to define the payload length in case of an implicit header. Is it necessary to use this register?
}

/*!
 * \brief TODO
 * No parameters are set here because a listener must be configured.
 * When the receive interrupt triggers, it will pass the received payload.
 * This section only sets the radio to receive mode.
 */
void loraRadio_receive(uint8_t len)
{
	uint8_t sdcfg0r_reg;
	uint8_t agccfg_reg;
	uint8_t liqpolr_reg;

	// TODO: Currently, no actions are taken within the interrupt handlers
	// because all interrupts are enabled during initialization.
	// If changed to an initialization where interrupts are not enabled by default, update here accordingly.

	loraRadio_setStandbyCommand(STDBY_RC);

	// Set RF switch to RX
	rfSwitch_setState(SWITCH_RX);

	// Set rx power mode
	loraRadio_writeRegister(SUBGHZ_RXGAINCR, (uint8_t)loraRadio_rxParameters.rx_power_mode);

	/*ST "SubGHz_Phy" example software WORKAROUND - Modulation Quality with 500 kHz LoRaTM Bandwidth*/
	// TODO: Ensure that this workaround is implemented in RX mode,
	// even though it appears in the TX function in the example. Confirm correct placement.

	sdcfg0r_reg = loraRadio_readRegister(SUBGHZ_SDCFG0R);

	if (loraRadio_rxParameters.modulation_parameters.bandwidth == LORA_BW_500)
	{
		sdcfg0r_reg &= ~(1 << 2);
		loraRadio_writeRegister(SUBGHZ_SDCFG0R, sdcfg0r_reg);
	}
	else
	{
		sdcfg0r_reg |= (1 << 2);
		loraRadio_writeRegister(SUBGHZ_SDCFG0R, sdcfg0r_reg);
	}

	// Set modulation parameters
	loraRadio_setRfFrequencyCommand(loraRadio_rxParameters.modulation_parameters.channel_freq);
	loraRadio_setModulationParamsCommand(loraRadio_rxParameters.modulation_parameters.spreading_factor,
																			 loraRadio_rxParameters.modulation_parameters.bandwidth,
																			 loraRadio_rxParameters.modulation_parameters.coding_rate,
																			 loraRadio_rxParameters.modulation_parameters.low_data_rate_opt);

	// With explicit header we don't know the payload length in advance (is variable). So we set max payload length --> 255
	if (loraRadio_rxParameters.packet_parameters.header_type == EXPLICIT_HEADER)
	{
		len = 0xFF;
	}

	// Set packet parameters
	loraRadio_setPacketParamsCommand(loraRadio_rxParameters.packet_parameters.preamble_len,
																	 loraRadio_rxParameters.packet_parameters.header_type,
																	 len,
																	 loraRadio_rxParameters.packet_parameters.crc_mode,
																	 loraRadio_rxParameters.packet_parameters.iq_mode);

	// Set symbol timeout
	if ((loraRadio_rxParameters.rx_mode == RX_CONTINUOUS_MODE))
	{
		loraRadio_setLoraSymbTimeoutCommand((uint8_t)0);
	}
	else
	{
		loraRadio_setLoraSymbTimeoutCommand(loraRadio_rxParameters.symbol_timeout);
	}

	/*ST "SubGHz_Phy" example software WORKAROUND - Set the step threshold value to 1 to avoid to miss low power signal after an interferer jam the chip in LoRa modulaltion */
	agccfg_reg = loraRadio_readRegister(SUBGHZ_AGCCFG);
	agccfg_reg &= 0x01;
	loraRadio_writeRegister(SUBGHZ_AGCCFG, agccfg_reg);

	/*ST "SubGHz_Phy" example software WORKAROUND - Optimizing the Inverted IQ Operation, see STM32WL Erratasheet */
	liqpolr_reg = loraRadio_readRegister(SUBGHZ_LIQPOLRL);

	if (loraRadio_rxParameters.packet_parameters.iq_mode == INVERTED_IQ)
	{
		liqpolr_reg &= ~(1 << 2);
		loraRadio_writeRegister(SUBGHZ_LIQPOLRL, liqpolr_reg);
	}
	else
	{
		liqpolr_reg |= (1 << 2);
		loraRadio_writeRegister(SUBGHZ_LIQPOLRL, liqpolr_reg);
	}

	// Set the configured rx mode
	if (loraRadio_rxParameters.rx_mode == RX_LISTENING_MODE)
	{
		// See STM32WL Errata: RadioSetRxDutyCycle
		loraRadio_rxParameters.rx_duty_cycle_preamble_detect_timeout =
				2 * loraRadio_rxParameters.listening_mode_rx_time +
				loraRadio_rxParameters.listening_mode_sleep_time;

		loraRadio_setRxDutyCycleCommand(loraRadio_rxParameters.listening_mode_rx_time,
																		loraRadio_rxParameters.listening_mode_sleep_time);
	}
	else
	{
		// See STM32WL Errata: RadioSetRxDutyCycle
		loraRadio_rxParameters.rx_duty_cycle_preamble_detect_timeout = 0;

		if (loraRadio_rxParameters.rx_mode == RX_CONTINUOUS_MODE)
		{
			loraRadio_setRxCommand((uint32_t)0xFFFFFF); // 0xFFFFFF == rx continuous. See Set_Rx() command in reference manual
		}
		else if (loraRadio_rxParameters.rx_mode == RX_SINGLE_MODE)
		{
			loraRadio_setRxCommand((uint32_t)0); // 0 == rx single mode without timeout. See Set_Rx() command in reference manual
		}
		else //(loraRadio_rxParameters.rx_mode == RX_SINGLE_WITH_TIMEOUT_MODE)
		{
			loraRadio_setRxCommand(loraRadio_rxParameters.rx_timeout);
		}
	}

	// TODO: Add a static variable in the module to track the current state.
	// Alternatively, consider implementing this module without its own state and use a "get State" command for the SubGhz module instead.
}

/*!
 * \brief TODO
 */
void loraRadio_startTxContinuousWave(uint32_t channel_freq, int8_t power_dBm)
{
	loraRadio_setStandbyCommand(STDBY_RC);

	if (power_dBm > 15) // Use HP PA
	{
		// Set RF switch to RFO HP
		rfSwitch_setState(SWITCH_RFO_HP);

		// Set tx power
		loraRadio_setTxParamsCommand(power_dBm + HP_POWER_OFFSET_dBm, RAMP_200_us);

		// set power amplifier to HP PA mode
		loraRadio_setPaConfigCommand(HP_PA_DUTY_CYCLE, HP_HP_MAX, HP_PA_SEL);
		loraRadio_writeRegister(SUBGHZ_PAOCPR, HP_OVER_CURRENT_PROTECTION_VALUE_FOR_PAOCPR_REGISTER);
	}
	else // Use LP PA
	{
		// Set RF switch to RFO LP
		rfSwitch_setState(SWITCH_RFO_LP);

		// Set tx power
		loraRadio_setTxParamsCommand(power_dBm + LP_POWER_OFFSET_dBm, RAMP_200_us);

		// set power amplifier to LP PA mode
		loraRadio_setPaConfigCommand(LP_PA_DUTY_CYCLE, LP_HP_MAX, LP_PA_SEL);
		loraRadio_writeRegister(SUBGHZ_PAOCPR, LP_OVER_CURRENT_PROTECTION_VALUE_FOR_PAOCPR_REGISTER);
	}

	// Set channel frequency
	loraRadio_setRfFrequencyCommand(channel_freq);

	loraRadio_setTxContinuousWaveCommand();

	// TODO: Add a static variable in the module to track the current state.
	// Alternatively, consider implementing this module without its own state and use a "get State" command for the SubGhz module instead.
}

/*!
 * \brief TODO
 */
void loraRadio_stopTxContinuousWave(void)
{
	loraRadio_setSleepCommand();
	// TODO: Add a static variable in the module to track the current state.
	// Alternatively, consider implementing this module without its own state and use a "get State" command for the SubGhz module instead.
}

// TODO: Review all LoRa erratas

// TODO: Implement functions for CAD (Channel Activity Detection), "is Channel Free", ...
